"use client"
//app/search/page.tsx
import { useState, useEffect, Suspense, useRef } from "react"
import { useSearchParams, useRouter } from "next/navigation"
import { Filter, Calendar, Shield, ImageIcon, VideoIcon, GlobeIcon, ChevronDownIcon } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { LogoWithText } from "@/components/logo"
import { SearchWithAutocomplete } from "@/components/search-with-autocomplete"
import { ImageModal } from "@/components/image-modal"
import { VideoModal } from "@/components/video-modal"
import { SettingsModal } from "@/components/settings-modal"
import { AdCard } from "@/components/ad-card"
import { useSettings } from "@/components/settings-provider"

interface SearchResult {
  id: string
  title: string
  description: string
  url: string
  favicon?: string
  date?: string
  keywords?: string[]
  safesearch: boolean
  maindomain: boolean
  isAd?: boolean
  // Ad-specific properties
  targetUrl?: string
  company?: string
  adType?: string
}

interface ImageResult {
  id: string
  title: string
  url: string
  imgurl: string
  keywords?: string[]
  favicon?: string
  date?: string
  safesearch: boolean
}

interface VideoResult {
  id: string
  title: string
  thumbnail: string
  url: string
  date?: string
  safesearch: boolean
  channelTitle?: string
  description?: string
}

function SearchPageContent() {
  const searchParams = useSearchParams()
  const router = useRouter()
  const { resultsPerPage, safeSearch: defaultSafeSearch, language, region } = useSettings()
  const isMountedRef = useRef(true)
  const [query, setQuery] = useState(searchParams.get("q") || "")
  const [searchType, setSearchType] = useState(searchParams.get("type") || "web")
  const [safeSearch, setSafeSearch] = useState(defaultSafeSearch ? "on" : "off")
  const [dateFilter, setDateFilter] = useState("all")
  const [results, setResults] = useState<SearchResult[]>([])
  const [imageResults, setImageResults] = useState<ImageResult[]>([])
  const [videoResults, setVideoResults] = useState<VideoResult[]>([])
  const [loading, setLoading] = useState(false)
  const [showFilters, setShowFilters] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const [selectedImage, setSelectedImage] = useState<ImageResult | null>(null)
  const [isModalOpen, setIsModalOpen] = useState(false)
  const [selectedVideo, setSelectedVideo] = useState<VideoResult | null>(null)
  const [isVideoModalOpen, setIsVideoModalOpen] = useState(false)
  const [currentPage, setCurrentPage] = useState(1)
  const [hasMoreResults, setHasMoreResults] = useState(false)
  const [totalResults, setTotalResults] = useState(0)

  useEffect(() => {
    isMountedRef.current = true

    const q = searchParams.get("q")
    const type = searchParams.get("type") || "web"
    if (q) {
      setQuery(q)
      setSearchType(type)
      performSearch(q, type)
    }

    return () => {
      isMountedRef.current = false
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [searchParams])

  useEffect(() => {
    if (isMountedRef.current) {
      setSafeSearch(defaultSafeSearch ? "on" : "off")
    }
  }, [defaultSafeSearch])

  const performSearch = async (searchQuery: string, type: string, page = 1) => {
    if (!isMountedRef.current) return

    setLoading(true)
    setError(null)
    try {
      console.log("[v0] Performing search:", {
        searchQuery,
        type,
        page,
        currentResults: results.length,
        currentImageResults: imageResults.length,
        currentVideoResults: videoResults.length,
      })

      const response = await fetch(
        `/api/search?q=${encodeURIComponent(searchQuery)}&type=${type}&safesearch=${safeSearch}&date=${dateFilter}&page=${page}&limit=${resultsPerPage}&lang=${language}&region=${region}`,
      )

      if (!response.ok) {
        throw new Error(`Search failed: ${response.status}`)
      }

      const data = await response.json()

      if (!isMountedRef.current) return

      console.log("[v0] Search response:", {
        page,
        newResultsCount: data.results?.length,
        hasMore: data.hasMore,
        total: data.total,
        adsCount: data.adsCount,
      })

      const newResults = data.results || []
      setTotalResults(data.total || newResults.length)
      setHasMoreResults(data.hasMore || false)

      if (type === "web") {
        if (page === 1) {
          setResults(newResults)
        } else {
          setResults((prevResults) => {
            console.log("[v0] Appending web results:", { prevCount: prevResults.length, newCount: newResults.length })
            return [...prevResults, ...newResults]
          })
        }
        setImageResults([])
        setVideoResults([])
      } else if (type === "images") {
        if (page === 1) {
          setImageResults(newResults)
        } else {
          setImageResults((prevResults) => {
            console.log("[v0] Appending image results:", { prevCount: prevResults.length, newCount: newResults.length })
            return [...prevResults, ...newResults]
          })
        }
        setResults([])
        setVideoResults([])
      } else if (type === "videos") {
        if (page === 1) {
          setVideoResults(newResults)
        } else {
          setVideoResults((prevResults) => {
            console.log("[v0] Appending video results:", { prevCount: prevResults.length, newCount: newResults.length })
            return [...prevResults, ...newResults]
          })
        }
        setResults([])
        setImageResults([])
      }

      setCurrentPage(page)
    } catch (error) {
      console.error("Search error:", error)
      if (isMountedRef.current) {
        setError(error instanceof Error ? error.message : "Search failed")
        if (page === 1) {
          setResults([])
          setImageResults([])
          setVideoResults([])
        }
      }
    } finally {
      if (isMountedRef.current) {
        setLoading(false)
      }
    }
  }

  const handleSearch = (searchQuery: string) => {
    if (searchQuery.trim()) {
      setCurrentPage(1)
      router.push(`/search?q=${encodeURIComponent(searchQuery.trim())}&type=${searchType}`)
    }
  }

  const handleTypeChange = (type: string) => {
    setSearchType(type)
    setCurrentPage(1)
    router.push(`/search?q=${encodeURIComponent(query)}&type=${type}`)
  }

  const handleSafeSearchChange = (value: string) => {
    setSafeSearch(value)
    setCurrentPage(1)
    if (query) {
      performSearch(query, searchType, 1)
    }
  }

  const handleDateFilterChange = (value: string) => {
    setDateFilter(value)
    setCurrentPage(1)
    if (query) {
      performSearch(query, searchType, 1)
    }
  }

  const handleLoadMore = () => {
    if (query && hasMoreResults && !loading && isMountedRef.current) {
      performSearch(query, searchType, currentPage + 1)
    }
  }

  const handleImageClick = (image: ImageResult) => {
    setSelectedImage(image)
    setIsModalOpen(true)
  }

  const handleCloseModal = () => {
    setIsModalOpen(false)
    setSelectedImage(null)
  }

  const handleVideoClick = (video: VideoResult) => {
    setSelectedVideo(video)
    setIsVideoModalOpen(true)
  }

  const handleCloseVideoModal = () => {
    setIsVideoModalOpen(false)
    setSelectedVideo(null)
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b border-border bg-card">
        <div className="max-w-7xl mx-auto px-4 py-3">
          {/* Top row: logo + search (stack on mobile) */}
          <div className="flex flex-col md:flex-row md:items-center gap-3">
            {/* Logo */}
            <div className="flex-shrink-0 flex items-center justify-start">
              <LogoWithText size="sm" />
            </div>

            {/* Search Form */}
            <div className="w-full md:flex-1 md:max-w-2xl">
              <SearchWithAutocomplete
                value={query}
                onChange={setQuery}
                onSubmit={handleSearch}
                placeholder="Search..."
                size="sm"
                showButton={true}
                buttonText="Search"
                className="!placeholder:text-black dark:!placeholder:text-white text-black dark:text-white w-full"
              />
            </div>
          </div>

          {/* Compact menu row: types and actions */}
          <div className="mt-3">
            {/* Type buttons: compact, small font, horizontally scrollable on very small screens */}
            <div className="overflow-x-auto -mx-4 px-4">
              <div className="inline-flex gap-2 w-full">
                <button
                  onClick={() => handleTypeChange("web")}
                  className={
                    `flex items-center gap-2 px-3 py-1 rounded-md min-w-[82px] justify-center text-xs font-medium ` +
                    (searchType === "web"
                      ? "bg-green-600 text-white hover:bg-green-700"
                      : "bg-transparent text-muted-foreground hover:bg-muted/40")
                  }
                >
                  <GlobeIcon className="w-4 h-4" />
                  <span className="truncate">Web</span>
                </button>

                <button
                  onClick={() => handleTypeChange("images")}
                  className={
                    `flex items-center gap-2 px-3 py-1 rounded-md min-w-[92px] justify-center text-xs font-medium ` +
                    (searchType === "images"
                      ? "bg-green-600 text-white hover:bg-green-700"
                      : "bg-transparent text-muted-foreground hover:bg-muted/40")
                  }
                >
                  <ImageIcon className="w-4 h-4" />
                  <span className="truncate">Images</span>
                </button>

                <button
                  onClick={() => handleTypeChange("videos")}
                  className={
                    `flex items-center gap-2 px-3 py-1 rounded-md min-w-[92px] justify-center text-xs font-medium ` +
                    (searchType === "videos"
                      ? "bg-green-600 text-white hover:bg-green-700"
                      : "bg-transparent text-muted-foreground hover:bg-muted/40")
                  }
                >
                  <VideoIcon className="w-4 h-4" />
                  <span className="truncate">Videos</span>
                </button>

                {/* Filters toggle compact */}
                <button
                  onClick={() => setShowFilters(!showFilters)}
                  className="flex items-center gap-2 px-3 py-1 rounded-md min-w-[88px] justify-center text-xs font-medium bg-transparent text-muted-foreground hover:bg-muted/40"
                >
                  <Filter className="w-4 h-4" />
                  <span className="truncate">Filters</span>
                  <ChevronDownIcon className={`w-3 h-3 transition-transform ${showFilters ? "rotate-180" : ""}`} />
                </button>

                {/* Settings: wrap SettingsModal in a compact button container for consistent sizing */}
                <div className="flex items-center">
                  <div className="inline-block">
                    <SettingsModal />
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Filters panel */}
          {showFilters && (
            <div className="flex flex-col sm:flex-row items-start sm:items-center gap-3 mt-3 p-3 bg-muted rounded-lg">
              <div className="flex items-center gap-2 w-full sm:w-auto">
                <Shield className="w-4 h-4 text-muted-foreground" />
                <span className="text-xs font-medium">Safe</span>
                <Select value={safeSearch} onValueChange={handleSafeSearchChange}>
                  <SelectTrigger className="w-20 h-7 text-xs">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="on">On</SelectItem>
                    <SelectItem value="off">Off</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="flex items-center gap-2 w-full sm:w-auto">
                <Calendar className="w-4 h-4 text-muted-foreground" />
                <span className="text-xs font-medium">Date</span>
                <Select value={dateFilter} onValueChange={handleDateFilterChange}>
                  <SelectTrigger className="w-28 h-7 text-xs">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All time</SelectItem>
                    <SelectItem value="24h">24h</SelectItem>
                    <SelectItem value="week">Week</SelectItem>
                    <SelectItem value="month">Month</SelectItem>
                    <SelectItem value="year">Year</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          )}
        </div>
      </header>

      {/* Results */}
      <main className="max-w-7xl mx-auto px-4 py-6">
        {loading ? (
          <div className="flex items-center justify-center py-12">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-accent"></div>
          </div>
        ) : error ? (
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <p className="text-muted-foreground mb-2">Search failed</p>
              <p className="text-sm text-muted-foreground">{error}</p>
              <Button
                variant="outline"
                className="mt-4 bg-transparent"
                onClick={() => query && performSearch(query, searchType)}
              >
                Try Again
              </Button>
            </div>
          </div>
        ) : (
          <>
            {/* === Web Results === */}
            {searchType === "web" && results.length === 0 && query && (
              <div className="text-center py-12">
                <p className="text-muted-foreground">No web results found for "{query}"</p>
                <img src="/no_result.png" alt="no_results" width={250} className="mx-auto" />
              </div>
            )}

            {searchType === "web" && results.length > 0 && (
              <div className="space-y-1 search-results-container">
                <div className="text-sm text-muted-foreground mb-6">About {totalResults.toLocaleString()} results</div>

                {results.map((result, index) => {
                  // Check if this is an ad
                  if (result.isAd) {
                    return (
                      <div key={`ad-${result.id}`} className="mb-8">
                        <AdCard ad={result} />
                      </div>
                    )
                  }

                  // Regular search result
                  return (
                    <div key={result.id} className="mb-8 w-full">
                      <div className="flex items-start gap-3 mb-1">
                        <div className="flex-shrink-0 mt-1">
                          <img
                            src={
                              result.favicon ||
                              `https://www.google.com/s2/favicons?domain=${new URL(result.url).hostname || "/placeholder.svg"}&sz=20`
                            }
                            alt=""
                            className="w-5 h-5 rounded-sm"
                            onError={(e) => {
                              e.currentTarget.src = "/placeholder.svg?height=20&width=20"
                            }}
                          />
                        </div>

                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 mb-1">
                            <span className="text-sm font-medium text-foreground truncate">
                              {new URL(result.url).hostname}
                            </span>
                            {result.maindomain && <span className="text-xs text-muted-foreground">• Official</span>}
                          </div>
                          <div className="text-sm text-green-700 dark:text-green-400 mb-1 truncate">{result.url}</div>
                        </div>
                      </div>

                      <h3 className="mb-2">
                        <a
                          href={result.url}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="text-lg sm:text-xl text-blue-600 dark:text-blue-400 hover:underline font-normal leading-tight"
                        >
                          {result.title}
                        </a>
                      </h3>

                      <p className="text-sm text-foreground leading-relaxed mb-2 max-w-full">{result.description}</p>

                      {result.keywords && result.keywords.length > 0 && (
                        <div className="flex flex-wrap gap-3 text-sm">
                          {result.keywords.slice(0, 4).map((keyword, index) => (
                            <a
                              key={keyword}
                              href={`/search?q=${encodeURIComponent(keyword)}&type=web`}
                              className="text-blue-600 dark:text-blue-400 hover:underline capitalize text-xs"
                            >
                              {keyword}
                            </a>
                          ))}
                        </div>
                      )}

                      {result.date && (
                        <div className="text-xs text-muted-foreground mt-2">
                          {new Date(result.date).toLocaleDateString("en-US", {
                            year: "numeric",
                            month: "short",
                            day: "numeric",
                          })}
                        </div>
                      )}
                    </div>
                  )
                })}

                {hasMoreResults && (
                  <div className="flex justify-center mt-8">
                    <Button onClick={handleLoadMore} disabled={loading} variant="outline" size="lg">
                      {loading ? "Loading..." : "Load More Results"}
                    </Button>
                  </div>
                )}
              </div>
            )}

            {/* === Image Results === */}
            {searchType === "images" && imageResults.length === 0 && query && (
              <div className="text-center py-12">
                <p className="text-muted-foreground">No image results found for "{query}"</p>
                <img src="/no_result.png" alt="no_results" width={250} className="mx-auto" />
              </div>
            )}

            {searchType === "images" && imageResults.length > 0 && (
              <>
                <div className="text-sm text-muted-foreground mb-6">About {totalResults.toLocaleString()} results</div>

                <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-4">
                  {imageResults.map((image) => (
                    <div key={image.id} className="group cursor-pointer" onClick={() => handleImageClick(image)}>
                      <div className="aspect-square bg-muted rounded-lg overflow-hidden mb-2 hover:shadow-md transition-shadow">
                        <img
                          src={image.imgurl || "/placeholder.svg"}
                          alt={image.title}
                          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-200"
                          onError={(e) => {
                            console.log("[v0] Image failed to load:", image.imgurl, "for", image.title)
                            if (image.url && !e.currentTarget.src.includes("google.com/s2/favicons")) {
                              try {
                                const domain = new URL(image.url).hostname
                                e.currentTarget.src = `https://www.google.com/s2/favicons?domain=${domain}&sz=128`
                              } catch {
                                e.currentTarget.src =
                                  "/placeholder.svg?height=200&width=200&text=" +
                                  encodeURIComponent(image.title.substring(0, 20))
                              }
                            } else {
                              e.currentTarget.src =
                                "/placeholder.svg?height=200&width=200&text=" +
                                encodeURIComponent(image.title.substring(0, 20))
                            }
                          }}
                          onLoad={() => {
                            console.log("[v0] Image loaded successfully:", image.imgurl, "for", image.title)
                          }}
                        />
                      </div>

                      <h3 className="text-sm font-medium line-clamp-2 mb-1 group-hover:text-blue-600 transition-colors">
                        {image.title}
                      </h3>

                      <div className="flex items-center gap-1 mb-2">
                        {image.favicon && (
                          <img
                            src={
                              image.favicon ||
                              `https://www.google.com/s2/favicons?domain=${new URL(image.url).hostname || "/placeholder.svg"}&sz=16`
                            }
                            alt=""
                            className="w-3 h-3 flex-shrink-0"
                            onError={(e) => {
                              e.currentTarget.src = "/placeholder.svg?height=12&width=12"
                            }}
                          />
                        )}
                        <span className="text-xs text-muted-foreground truncate">{new URL(image.url).hostname}</span>
                      </div>

                      {image.keywords && image.keywords.length > 0 && (
                        <div className="flex flex-wrap gap-1">
                          {image.keywords.slice(0, 3).map((keyword) => (
                            <span
                              key={keyword}
                              className="text-xs text-blue-600 hover:underline cursor-pointer"
                              onClick={(e) => {
                                e.stopPropagation()
                                router.push(`/search?q=${encodeURIComponent(keyword)}&type=images`)
                              }}
                            >
                              {keyword}
                            </span>
                          ))}
                        </div>
                      )}
                    </div>
                  ))}
                </div>

                {hasMoreResults && (
                  <div className="flex justify-center mt-8">
                    <Button onClick={handleLoadMore} disabled={loading} variant="outline" size="lg">
                      {loading ? "Loading..." : "Load More Images"}
                    </Button>
                  </div>
                )}

                <ImageModal image={selectedImage} isOpen={isModalOpen} onClose={handleCloseModal} />
              </>
            )}

            {/* === Video Results === */}
            {searchType === "videos" && videoResults.length === 0 && query && (
              <div className="text-center py-12">
                <p className="text-muted-foreground">No video results found for "{query}"</p>
                <p className="text-sm text-muted-foreground mt-2">
                  Make sure your YouTube API key is configured in environment variables
                </p>
              </div>
            )}

            {searchType === "videos" && videoResults.length > 0 && (
              <>
                <div className="text-sm text-muted-foreground mb-6">About {totalResults.toLocaleString()} results</div>

                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                  {videoResults.map((video) => (
                    <Card
                      key={video.id}
                      className="group overflow-hidden hover:shadow-xl transition-all duration-300 border-0 bg-card/50 backdrop-blur-sm cursor-pointer"
                      onClick={() => handleVideoClick(video)}
                    >
                      <div className="aspect-video relative overflow-hidden">
                        <img
                          src={video.thumbnail || "/placeholder.svg"}
                          alt={video.title}
                          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                        />
                        <div className="absolute inset-0 flex items-center justify-center bg-black/40 opacity-0 group-hover:opacity-100 transition-all duration-300">
                          <div className="w-16 h-16 bg-red-600 rounded-full flex items-center justify-center shadow-lg transform scale-75 group-hover:scale-100 transition-transform duration-300">
                            <svg className="w-8 h-8 text-white ml-1" fill="currentColor" viewBox="0 0 24 24">
                              <path d="M8 5v14l11-7z" />
                            </svg>
                          </div>
                        </div>
                        <div className="absolute bottom-2 right-2 bg-black/80 text-white text-xs px-2 py-1 rounded">
                          Video
                        </div>
                      </div>
                      <CardContent className="p-5">
                        <h3 className="font-semibold mb-3 line-clamp-2 text-primary leading-snug">{video.title}</h3>
                        <div className="space-y-2">
                          {video.channelTitle && (
                            <p className="font-medium text-accent text-sm">{video.channelTitle}</p>
                          )}
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-1 text-xs text-muted-foreground">
                              <Calendar className="w-3 h-3" />
                              <span>{video.date ? new Date(video.date).toLocaleDateString() : ""}</span>
                            </div>
                            <span className="inline-flex items-center gap-1 text-accent hover:text-accent/80 font-medium text-sm transition-colors group-hover:underline decoration-2 underline-offset-2">
                              <VideoIcon className="w-3 h-3" />
                              Watch
                            </span>
                          </div>
                        </div>
                        {video.description && (
                          <p className="text-sm text-foreground/70 mt-3 line-clamp-2 leading-relaxed">
                            {video.description}
                          </p>
                        )}
                      </CardContent>
                    </Card>
                  ))}
                </div>

                {hasMoreResults && (
                  <div className="flex justify-center mt-8">
                    <Button onClick={handleLoadMore} disabled={loading} variant="outline" size="lg">
                      {loading ? "Loading..." : "Load More Videos"}
                    </Button>
                  </div>
                )}

                <VideoModal video={selectedVideo} isOpen={isVideoModalOpen} onClose={handleCloseVideoModal} />
              </>
            )}
          </>
        )}
      </main>
    </div>
  )
}

export default function SearchPage() {
  return (
    <Suspense fallback={<div>Loading...</div>}>
      <SearchPageContent />
    </Suspense>
  )
}
